/**
* \file: Session.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <linux/input.h>
#include <adit_logging.h>
#include <fstream>
#include <string.h>
#include <unistd.h>
#include "Session.h"
#include <aauto/AoapTransport.h>


#ifdef AAUTO_AUTHENTICATION_SDC
#include <aauto/SdcAuthenticator.h>
#else
#include <aauto/DevelopersAuthenticator.h>
#endif

LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {

static std::string aautoDemoGetId();

std::string aautoDemoGetId() {
    return "8d11fb57-c7cb-44c6-8599-19d763969437";
}

Session::Session(IDynamicConfiguration& inConfig, const std::string inSerial)
{
    LOGD_DEBUG((demo, "session created"));
    // Create GalReceiver object
    receiver = ::shared_ptr<GalReceiver>(new GalReceiver());

    // Create GalReceiver IControllerCallbacks object
    // TODO actually Session should implement IControllerCallbacks
    callbacks = ::shared_ptr<AautoDemoIControllerCallbacks>(new AautoDemoIControllerCallbacks(receiver, inSerial));
    transport = nullptr;
    mConfig   = &inConfig;
    mUsbKeyboard = (bool)mConfig->GetNumber("aauto-demo-usb-keyboard", 0);
}

Session::~Session()
{
    Stop();

    receiver->shutdown();

    /* drop the reference to the GalReceiver to solve cross dependency
     * between GalReceiver and AautoDemoIControllerCallbacks */
    callbacks->releaseReceiver();
    callbacks = nullptr;
    transport = nullptr;
    receiver = nullptr;

    LOG_INFO((demo, "session destroyed"));
}

bool Session::attachTransport(::shared_ptr<Transport> inTransport)
{
    if(inTransport == nullptr)
    {
        LOG_INFO((demo, "attachTransport parameter is null."));
        return false;
    }
    
    transport = inTransport;
    transport->registerCallbacks(callbacks);
    return true;
}

bool Session::Start()
{
    int id = 1; 
    
    if (!receiver->init(callbacks))
    {
        LOG_ERROR((demo, "Could not init GalReceiver"));
        // TODO error handling
        return false;
    }

    if (!initReceiver())
    {
        LOG_ERROR((demo, "Could not init GalReceiver"));
        // TODO error handling
        return false;
    }

    // start video
    /* video endpoint always has to be created before input,
     * in case they are sharing same surface / wl_display */
    video = new VideoChannel(receiver, id++, true, layerManager->getLayerId(),\
                            layerManager->getSurfaceId(serial), *mConfig);
    if (!video->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize video channel"));
        // don't cancel session start
    }

    // start inputsource
    /* video endpoint always has to be created before input,
     * in case they are sharing same surface / wl_display */
    input = new InputSourceChannel(receiver, id++, layerManager->getLayerId(), layerManager->getTouchSurfaceId(serial), *mConfig);
    if (!input->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize input source channel"));
        // don't cancel session start
    }

    // start audio sink media
    mediaAudio = new AudioSinkChannel(receiver, id++, AUDIO_STREAM_MEDIA, *mConfig);
    if (!mediaAudio->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize audio sink media channel"));
        // don't cancel session start
    }

    // start audio sink system
    systemAudio = new AudioSinkChannel(receiver, id++, AUDIO_STREAM_SYSTEM_AUDIO, *mConfig);
    if (!systemAudio->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize audio sink system channel"));
        // don't cancel session start
    }

    // start audio sink navi
    navAudio = new AudioSinkChannel(receiver, id++, AUDIO_STREAM_GUIDANCE, *mConfig);
    if (!navAudio->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize audio sink navi channel"));
        // don't cancel session start
    }

    // start audio source
    micAudio = new AudioSourceChannel(receiver, id++, *mConfig);
    if (!micAudio->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize audio source channel"));
        // don't cancel session start
    }

    // HU should report what transformations it performs on GPS data using 'locationCharacterization' flags.
    // In the example code snippet of Google, they set it to 0.
    uint32_t locationCharacterization = 0;
    // start sensor source
    sensor = new SensorSourceChannel(receiver, id++, locationCharacterization);
    if (!sensor->Initialize(testParameter.getLocation()))
    {
        LOG_ERROR((demo, "Could not initialize sensor source channel"));
        // don't cancel session start
    }
    
    // start navigation status
    navi = new NaviStatusChannel(receiver, id++, sensor.get());
    if (!navi->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize navigation status channel"));
        // don't cancel session start
    }

    // start media browse
    mediaBs = new MediaBrowserChannel(receiver, id++);
    if (!mediaBs->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize navigation status channel"));
        // don't cancel session start
    }

    // start media playback status
    mediaPb = new MediaPlaybackStatusChannel(receiver, id++);
    if (!mediaPb->Initialize())
    {
        LOG_ERROR((demo, "Could not initialize media playback status channel"));
        // don't cancel session start
    }

    if (mUsbKeyboard)
    {
        // start keyboard
        const struct eventKeyTblType {
            eventTypeCode   type;
            uint16_t        key;
            uint32_t        input;
        }evtKeyTbl[] = {
                            {InputEventType, KEY_A,     KEYCODE_MEDIA_PREVIOUS                           },
                            {InputEventType, KEY_S,     KEYCODE_MEDIA_PLAY                               },
                            {InputEventType, KEY_D,     KEYCODE_MEDIA_PAUSE                              },
                            {InputEventType, KEY_F,     KEYCODE_MEDIA_NEXT                               },
                            {InputEventType, KEY_V,     KEYCODE_MEDIA_RECORD                             },
                            {InputEventType, KEY_UP,    KEYCODE_DPAD_UP                                  },
                            {InputEventType, KEY_DOWN,  KEYCODE_DPAD_DOWN                                },
                            {InputEventType, KEY_LEFT,  KEYCODE_DPAD_LEFT                                },
                            {InputEventType, KEY_RIGHT, KEYCODE_DPAD_RIGHT                               },
                            {InputEventType, KEY_ENTER, KEYCODE_ENTER                                    },
                            {InputEventType, KEY_TAB,   KEYCODE_TAB                                      },
                            {InputEventType, KEY_Q,     KEYCODE_NAVIGATION                               },
                            {VideoEventType, KEY_N,     VIDEO_NATIVE                                     },
                            {VideoEventType, KEY_P,     VIDEO_PROJECTION                                 },
                            {DeviceSelectionType, KEY_1, DEVICE_SEL_1                                    },
                            {DeviceSelectionType, KEY_2, DEVICE_SEL_2                                    },
                            {DeviceSelectionType, KEY_3, DEVICE_SEL_3                                    },
                            {DeviceSelectionType, KEY_4, DEVICE_SEL_4                                    },
                            {DeviceSelectionType, KEY_5, DEVICE_SEL_5                                    },
                            {DeviceSelectionType, KEY_6, DEVICE_SEL_6                                    },
                            {DeviceSelectionType, KEY_7, DEVICE_SEL_7                                    },
                            {DeviceSelectionType, KEY_8, DEVICE_SEL_8                                    },
                            {DeviceSelectionType, KEY_9, DEVICE_SEL_9                                    },
                            {DeviceSelectionType, KEY_0, DEVICE_SEL_10                                   },
                            {MediaActionType, KEY_U, InstrumentClusterInput_InstrumentClusterAction_UP   },
                            {MediaActionType, KEY_M, InstrumentClusterInput_InstrumentClusterAction_DOWN },
                            {MediaActionType, KEY_H, InstrumentClusterInput_InstrumentClusterAction_LEFT },
                            {MediaActionType, KEY_K, InstrumentClusterInput_InstrumentClusterAction_RIGHT},
                            {MediaActionType, KEY_J, InstrumentClusterInput_InstrumentClusterAction_ENTER},
                            {MediaActionType, KEY_B, InstrumentClusterInput_InstrumentClusterAction_BACK },
                            {MediaActionType, KEY_C, InstrumentClusterInput_InstrumentClusterAction_CALL },
                            {MediaActionType, KEY_Z, MEDIA_BROWSE_GET_NODE                               }
                        };

        keyboard = new USBKeyboardInput();

        for(uint32_t i = 0; i < (sizeof(evtKeyTbl) / sizeof(eventKeyTblType)); i++)
        {
            keyboard->addSupportCode(evtKeyTbl[i].type, evtKeyTbl[i].key, evtKeyTbl[i].input);
        }

        ::shared_ptr<AditInputSource> inputSource = input->getProtocolEndpoint();
        if(inputSource != nullptr)
        {
            inputSource->registerKeycodes(keyboard->getSetKeycodes());
            if(!keyboard->init(this))
            {
                LOG_ERROR((demo, "Could not initialize USB Keyboard"));
            }
        }
        else
        {
            LOG_ERROR((demo, "Could not start USB Keyboard"));
        }
    }
    else
    {
        LOGD_DEBUG((demo, "USB Keyboard disabled (mUsbKeyboard = %d)", mUsbKeyboard));
    }

    receiver->start();
    // TODO error handling

    LOG_INFO((demo, "session started"));
    return true;
}

void Session::Stop()
{
    if (keyboard != nullptr){
        keyboard->shutdown();
    }
    if (sensor != nullptr){
        sensor->shutdown();
    }
    if (mediaAudio != nullptr){
        mediaAudio->shutdown();
    }
    if (systemAudio != nullptr){
        systemAudio->shutdown();
    }
    if (navAudio != nullptr){
        navAudio->shutdown();
    }
    if (micAudio != nullptr){
        micAudio->shutdown();
    }
    /* input endpoint always has to be shutdown before video,
     * in case they are sharing same surface / wl_display */
    if (input != nullptr){
        input->shutdown();
    }
    /* input endpoint always has to be shutdown before video,
     * in case they are sharing same surface / wl_display */
    if (video != nullptr){
        video->shutdown();
    }
    if (navi != nullptr){
        navi->shutdown();
    }
    if (mediaBs != nullptr){
        mediaBs->shutdown();
    }
    if (mediaPb != nullptr){
        mediaPb->shutdown();
    }
    LOG_INFO((demo, "session stopped"));
}

void Session::waitForExit()
{
    if(transport != nullptr)
    {
        transport->waitForExit();
    }
}

void Session::requestStop()
{
    if(transport != nullptr)
    {
        transport->requestStop();
    }
}

void Session::sendByeByeRequest()
{
    unsigned int waitResponse = 20;
    if(transport != nullptr)
    {
        LOG_INFO((demo, "Session::sendByeByeRequest()"));
        receiver->sendByeByeRequest(USER_SELECTION);
        /* wait for byebyeResponse from MD before continue */
        AautoDemoIControllerCallbacks* controller = (AautoDemoIControllerCallbacks*)callbacks.get();
        // TODO: Remvoe while-loop
        while ((waitResponse > 0) && (true != controller->getRecvByeByeResponse())) {
            usleep(100000);
            waitResponse--;
        }
    }
}

bool Session::initReceiver()
{
    /* Set the credentials used in the ssl handshake. */
#ifdef AAUTO_AUTHENTICATION_SDC
    SdcAuthenticator authenticator = SdcAuthenticator();
#else
    DevelopersAuthenticator authenticator = DevelopersAuthenticator();
#endif
    authenticator.setConfigItem("keyId", "3434");
    authenticator.setConfigItem("rootCertificate", "/opt/dev_certificates/wr_root_cert");
    authenticator.setConfigItem("clientCertificate", "/opt/dev_certificates/wr_client_cert");
    authenticator.setConfigItem("privateKey", "/opt/dev_certificates/wr_private_key");
    if(!authenticator.setCertificates(receiver))
    {
        LOG_ERROR((demo, "Authentication certificates could not be set"));
        return false;
    }
    else
    {
        LOG_INFO((demo, "Authentication certificates set successfully"));
    }

    /* Set up identity information of the car. */
    receiver->setIdentityInfo("ADIT", "ADIT super car", "2017", aautoDemoGetId().c_str());
    /* Sets the driver position - left/right/center.
     * This influences the layout of the screen */
    receiver->setDriverPosition(DRIVER_POSITION_LEFT);

    /* Our system does not run wall time.
     * We have to set the time that will be used to verify
     * the validity of the certificate presented by the phone.
     * @param:  The number of seconds elapsed since Jan 1 1970 UTC 00:00:00.*/
    receiver->setCertificateVerificationTime(1417449318);

    return true;
}

/*
 * KeyEventInput
 */
#define NSEC_PER_SEC    (1000ULL * 1000ULL * 1000ULL)
void Session::onKeyEventInput(uint32_t inputCode, bool inPressed)
{
    LOG_INFO((demo, "Key event input!(code:%d, press:%d) device %s", inputCode, inPressed, serial.c_str()));

    struct timespec tp;
    clock_gettime(CLOCK_MONOTONIC, &tp);
    uint64_t timestamp = (tp.tv_sec * NSEC_PER_SEC + tp.tv_nsec) / 1000;

    input->getProtocolEndpoint()->reportKey(timestamp, inputCode, inPressed, 0);
}

void Session::onKeyEventVideo(uint32_t inputCode)
{
    LOG_INFO((demo, "Key event video!(code:%d)", inputCode));

    ::shared_ptr<AditVideoSink> videoSink = video->getProtocolEndpoint();
    if(videoSink != nullptr)
    {
        if(inputCode == VIDEO_NATIVE)
        {
            videoSink->setVideoFocus(VIDEO_FOCUS_NATIVE, false);
        }
        else if(inputCode == VIDEO_PROJECTION)
        {
            videoSink->setVideoFocus(VIDEO_FOCUS_PROJECTED , false);
            keyboard->display();
        }
    }
}
void Session::onKeyEventDeviceSelection(uint32_t inputCode)
{
    LOG_INFO((demo, "Key event device selection (code:%d)", inputCode));

    if ((inputCode >= DEVICE_SEL_1) && (inputCode < DEVICE_SEL_MAX))
    {
        layerManager->bringDeviceToFront(inputCode - DEVICE_SEL_1);
    }
    else
    {
        LOG_INFO((demo, "error: illegal event code for device selection (%d)", inputCode));
        return;
    }
}

void Session::onKeyEventMediaAction(uint32_t inputCode)
{
    LOG_INFO((demo, "Key event media action (code:%d)", inputCode));
            
    if(inputCode == MEDIA_BROWSE_GET_NODE)
    {
        int ret;
        std::string path = "/";
        ret = mediaBs->mediaGetNode(path.c_str(), 1, false);
        LOG_INFO((demo, "mediaGetNode = %d path: %s", ret, path.c_str()));
        
    }
    else
    {
        mediaPb->mediaPlaybackReportAction(inputCode);
    }

}

} } /* namespace adit { namespace aauto { */
